package com.agilex.healthcare.directscheduling.resource;

import com.agilex.healthcare.directscheduling.dataservice.ConsolidatedService;
import com.agilex.healthcare.directscheduling.dataservice.DSDataService;
import com.agilex.healthcare.directscheduling.dataservice.DSFacilityProviderDataService;
import com.agilex.healthcare.directscheduling.dataservice.TeamListCodes;
import com.agilex.healthcare.directscheduling.domain.AppointmentTimeSlots;
import com.agilex.healthcare.directscheduling.domain.CancelAppointment;
import com.agilex.healthcare.directscheduling.domain.CancelReasons;
import com.agilex.healthcare.directscheduling.domain.FacilityMemberTeam;
import com.agilex.healthcare.directscheduling.domain.PatientRequestLimit;
import com.agilex.healthcare.directscheduling.domain.PatientVisited;
import com.agilex.healthcare.directscheduling.domain.VARBookedAppointmentCollections;
import com.agilex.healthcare.directscheduling.exception.AppointmentException;
import com.agilex.healthcare.directscheduling.linkbuilder.DirectSchedulingResourceDirectoryBuilder;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.ResourceDirectory;
import com.agilex.healthcare.veteranappointment.dataservice.AppointmentRequestDataService;
import com.agilex.healthcare.veteranappointment.dataservice.PatientMetadataDataService;
import com.agilex.healthcare.veteranappointment.resources.AbstractUserResource;
import com.agilex.healthcare.veteranappointment.utils.SessionStateHelper;
import com.google.common.collect.Lists;
import gov.va.vamf.mdws.client.v1.mobilehealthplatform.domain.MhpUser;
import gov.va.vamf.scheduling.communitycare.domain.BookedCCAppointment;
import gov.va.vamf.scheduling.communitycare.service.CCService;
import gov.va.vamf.scheduling.direct.InstitutionService;
import gov.va.vamf.scheduling.direct.datalayer.eligibility.ClinicalServiceService;
import gov.va.vamf.scheduling.direct.datalayer.facility.CustomFriendlyTextService;
import gov.va.vamf.scheduling.direct.domain.AppointmentEligibilityAtInstitution;
import gov.va.vamf.scheduling.direct.domain.CoreSettings;
import gov.va.vamf.scheduling.direct.domain.CustomFriendlyText;
import gov.va.vamf.scheduling.direct.domain.Institution;
import gov.va.vamf.scheduling.direct.domain.ValidationError;
import gov.va.vamf.scheduling.direct.eligibilitychecker.EligibilityCheckWorker;
import gov.va.vamf.security.v1.VamfJwtClaimsConstants;
import gov.va.vamf.security.v1.filters.JwtRbacRestricted;
import gov.va.vamf.security.v1.filters.JwtResourceRestricted;
import gov.va.vamf.videoconnect.common.domain.Appointment;
import gov.va.vamf.videoconnect.common.domain.AppointmentKind;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ExecutionException;

@Path("/direct-scheduling")
@Component
@Scope("request")
public class DSResource extends AbstractUserResource {

    private static final Logger logger = LoggerFactory.getLogger(DSResource.class);

    private static final String VAR_SOURCE_SYSTEM = "VAR";

    @Resource
    DSDataService dsDataService;

    @Resource
    private ConsolidatedService consolidatedService;

    @Resource
    DSFacilityProviderDataService dsFacilityProviderDataService;

    @Resource
    DirectSchedulingResourceDirectoryBuilder dsResourceDirectoryBuilder;

    @Resource
    SessionStateHelper sessionStateHelper;

    @Resource
    private InstitutionService institutionService;

    @Resource
    private ClinicalServiceService clinicalServiceService;

    @Resource
    private EligibilityCheckWorker eligibilityCheckWorker;

    @Resource
    private AppointmentRequestDataService appointmentRequestDataService;

    @Resource
    private CustomFriendlyTextService customFriendlyTextService;

    @Resource
    private CCService ccService;
    @Resource
    PatientMetadataDataService patientMetadataDataService;

    /**
     * This REST service is used to get resource directory for direct scheduling.
     *
     * @param uriInfo
     * @param siteCode
     * @return ResourceDirectory
     */
    @GET
    @Path("/directscheduling-resource-directory")
    @Produces({"application/xml", "application/json"})
    @JwtRbacRestricted("Veteran")
    public ResourceDirectory getResourceDirectory(@Context ContainerRequestContext context,
                                                  @Context UriInfo uriInfo,
                                                  @QueryParam("siteCode") String siteCode) {
        MhpUser user = getCurrentUser(context);
        ResourceDirectory rd = dsResourceDirectoryBuilder.getResourceDirectory(uriInfo.getBaseUri(), siteCode,
            user.getUserIdentifier().getAssigningAuthority(), user.getUserIdentifier().getUniqueId());
        return rd;
    }

    /**
     * This REST service is used to fetch all the appointment slots for selected facility from vista via scheduling manager for logged in patient/veteran
     *
     * @param uriInfo
     * @param request
     * @param headers
     * @param siteCode
     * @param assigningAuthority
     * @param patientId
     * @param startDate
     * @param endDate
     * @param clinicIds
     * @return Collection<AppointmentTimeSlots>
     */
    @GET
    @Path("/site/{site-code}/patient/{assigning-authority}/{patient-id}/available-appointment-slots")
    @Produces({"application/xml", "application/json"})
    @JwtResourceRestricted
    public Collection<AppointmentTimeSlots> getAppointmentSlots(@Context UriInfo uriInfo,
                                                                @Context HttpServletRequest request,
                                                                @Context HttpHeaders headers,
                                                                @PathParam("site-code") String siteCode,
                                                                @PathParam("assigning-authority") String assigningAuthority,
                                                                @PathParam("patient-id") String patientId,
                                                                @QueryParam("startDate") String startDate,
                                                                @QueryParam("endDate") String endDate,
                                                                @QueryParam("clinicIds") String clinicIds) throws InterruptedException, ExecutionException {

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        Collection<AppointmentTimeSlots> appointmentTimeSlots = dsDataService.getFilteredAppointmentSlots(uriInfo,
            patientIdentifier, siteCode, clinicIds);
        return appointmentTimeSlots;
    }

    /**
     * This REST service is used to update the booked appointment status to cancel in selected facility in vista via scheduling manager for logged in patient/veteran
     *
     * @param cancelAppointment
     * @param siteCode
     * @param assigningAuthority
     * @param patientId
     * @param uriInfo
     * @param headers
     * @param request
     * @return Response
     */
    @PUT
    @Path("/site/{site-code}/patient/{assigning-authority}/{patient-id}/cancel-appointment")
    @Produces({"application/xml", "application/json"})
    @Consumes({"application/json", "application/xml"})
    @JwtResourceRestricted
    public Response cancelPatientAppointment(CancelAppointment cancelAppointment,
                                             @PathParam("site-code") String siteCode,
                                             @PathParam("assigning-authority") String assigningAuthority,
                                             @PathParam("patient-id") String patientId,
                                             @Context UriInfo uriInfo,
                                             @Context HttpHeaders headers,
                                             @Context HttpServletRequest request) {
        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        Response response = dsDataService.cancelAppointmentForPatient(cancelAppointment, patientIdentifier, siteCode, request);
        return response;
    }

    /**
     * This REST service is used to save appointment for selected facility in vista via scheduling manager for logged in patient/veteran
     *
     * @param appointment
     * @param siteCode
     * @param assigningAuthority
     * @param patientId
     * @param request
     * @return Response
     */
    @POST
    @Path("/site/{site-code}/patient/{assigning-authority}/{patient-id}/booked-appointments")
    @Produces({"application/xml", "application/json"})
    @Consumes({"application/json", "application/xml"})
    @JwtResourceRestricted
    public Response bookNewAppointment(@Valid Appointment appointment,
                                       @Context ContainerRequestContext containerRequestContext,
                                       @PathParam("site-code") String siteCode,
                                       @PathParam("assigning-authority") String assigningAuthority,
                                       @PathParam("patient-id") String patientId,
                                       @Context HttpServletRequest request) {

        Response response;

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        appointment.setSourceSystem(VAR_SOURCE_SYSTEM);

        if (appointment.getAppointmentKind().equals(AppointmentKind.MOBILE_ANY)) {
            try {
                String jwtString = containerRequestContext.getHeaderString(VamfJwtClaimsConstants.VAMF_JWT_HEADER);
                Appointment bookedAppointment = consolidatedService.bookMobileAnyAppointment(jwtString, patientIdentifier, appointment);
                response = Response.ok(bookedAppointment).build();
            }
            catch(AppointmentException ex) {
                ValidationError error = ex.getError();
                int statusCode = Integer.parseInt(error.getErrorField());
                String logMsg = ex.getErrorMessageForLogging(statusCode);

                logger.debug(logMsg);

                return Response.status(statusCode).entity(error.getErrorMessage()).build();
            }
        }
        else {
            response = dsDataService.bookNewAppointmentForPatient(appointment, patientIdentifier, siteCode, request);
        }

        return response;
    }

    /**
     * This REST service is used to fetch cancel reason list for selected facility from vista via scheduling manager for logged in patient/veteran
     *
     * @param siteCode
     * @param uriInfo
     * @param headers
     * @param request
     * @return CancelReasons
     */
    @GET
    @Produces({"application/json", "application/xml"})
    @Path("/site/{site-code}/patient/{assigning-authority}/{patient-id}/cancel-reasons-list")
    @JwtResourceRestricted
    public CancelReasons getCancelReasonsDropDownList(@PathParam("site-code") String siteCode,
                                                      @Context UriInfo uriInfo,
                                                      @Context HttpHeaders headers,
                                                      @Context HttpServletRequest request,
                                                      @PathParam("assigning-authority") String assigningAuthority,
                                                      @PathParam("patient-id") String patientId) {
        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        CancelReasons cancelReasons = dsDataService.getCancelReasonList(patientIdentifier, siteCode);
        return cancelReasons;
    }

    /**
     * This REST service is used to fetch all booked appointments for selected facility from vista via scheduling manager for logged in patient/veteran.
     * Service returns collection with booked appointments by "PRIMART CARE" and "OTHER"
     *
     * @param uriInfo
     * @param siteCode
     * @param patientId
     * @param assigningAuthority
     * @param startDate
     * @param clinicIds
     * @return VARBookedAppointmentCollections
     */
    @GET
    @Produces({"application/xml", "application/json"})
    @Path("/site/{site-code}/patient/{assigning-authority}/{patient-id}/booked-appointments")
    @JwtResourceRestricted
    public VARBookedAppointmentCollections getPendingAppointmentsForPatient(@Context UriInfo uriInfo,
                                                                            @PathParam("site-code") String siteCode,
                                                                            @PathParam("patient-id") String patientId,
                                                                            @PathParam("assigning-authority") String assigningAuthority,
                                                                            @QueryParam("startDate") String startDate,
                                                                            @QueryParam("clinicIds") String clinicIds) {
        //VA Appointments
        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);
        VARBookedAppointmentCollections bookedAppointments = dsDataService.getPatientBookedAppointments(uriInfo, patientIdentifier, siteCode, clinicIds);
        return bookedAppointments;
    }

    /**
     * This REST service is used to fetch all booked community care appointments for a given patient/veteran.
     *
     * @param uriInfo
     * @param patientId
     * @param assigningAuthority
     * @param startDate
     * @return VARAppointmentRequests
     */
    @GET
    @Produces({"application/xml", "application/json"})
    @Path("/patient/{assigning-authority}/{patient-id}/booked-cc-appointments")
    @JwtResourceRestricted
    public VARBookedAppointmentCollections getPendingCCAppointmentsForPatient(@Context UriInfo uriInfo,
                                                                     @PathParam("patient-id") String patientId,
                                                                     @PathParam("assigning-authority") String assigningAuthority,
                                                                     @QueryParam("startDate") String startDate){
        //CC Booked Requests
        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        List<BookedCCAppointment> bookedCCAppointmentList = Lists.newArrayList(ccService.fetchCCBookedAppointmentsByPatientId(patientIdentifier));

        return dsDataService.getPatientCCBookedAppointments(bookedCCAppointmentList);
    }

    /**
     * This REST service is used to logout MDWS session open for selected facility for logged in patient/veteran
     *
     * @param uriInfo
     * @param request
     * @param headers
     * @param siteCode
     * @return Response
     */
    @GET
    @Produces({"application/xml", "application/json"})
    @Path("/site/{site-code}/logout")
    @JwtRbacRestricted("Veteran")
    public Response logout(@Context UriInfo uriInfo, @Context HttpServletRequest request, @Context HttpHeaders headers,
                           @PathParam("site-code") String siteCode) {
        return Response.ok().build();
    }

    /**
     * This REST service is used to retrieve all facilities by type of cares given the user authenticated parent site codes,
     * supported direct or request eligibility criteria or express care.
     *
     * @param facilityCodes the 3-digit facilities to fetch institutions for
     * @param parentCode an optional parent station to filter institutions by
     * @param clinicalServiceId a type of care/clinical service to filter institutions by
     * @return List of AppointmentEligibilityAtInstitution
     */
    @GET
    @Path("/institutions")
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    @JwtRbacRestricted("Veteran")
    public Collection<AppointmentEligibilityAtInstitution> getEligibleInstitutionsByTypeOfCare(
            @QueryParam("facility-code") final List<String> facilityCodes,
            @QueryParam("parent-code") final String parentCode,
            @NotNull @QueryParam("clinical-service") final String clinicalServiceId
    ) {
        final Collection<AppointmentEligibilityAtInstitution> appointmentEligibility =
                institutionService.getInstitutionsEligibleForClinicalService(
                        facilityCodes,
                        parentCode,
                        clinicalServiceId
                );
        return appointmentEligibility;
    }

    @GET
    @Path("/parent-sites")
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    @JwtRbacRestricted("Veteran")
    public Collection<Institution> getParentSites(
            @NotNull @QueryParam("facility-code") final List<String> facilityCodes
    ) {
        Collection<Institution> institutions = institutionService.getParentInstitutions(facilityCodes);
        return institutions;
    }

    /**
     * This REST service is used to verify that a patient has visited a clinic by type of care in the past configured direct eligibility days.
     *
     * @param siteCode
     * @param patientId
     * @param assigningAuthority
     * @param clinicalServiceId
     * @return PatientVisited
     */
    @GET
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    @Path("/site/{site-code}/patient/{assigning-authority}/{patient-id}/direct-eligibility/visited-in-past-months")
    @JwtResourceRestricted
    public PatientVisited getVisitedInPastMonthsByTypeOfCareDirectEligiblity(@NotNull @PathParam("site-code") String siteCode,
                                                                            @NotNull @PathParam("assigning-authority") String assigningAuthority,
                                                                            @NotNull @PathParam("patient-id") String patientId,
                                                                            @NotNull @QueryParam("clinical-service") final String clinicalServiceId,
                                                                            @NotNull @QueryParam("institution-code") final String institutionCode) {

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        return eligibilityCheckWorker.hasVisitedClinicInPastMonthsByTypeOfCare(patientIdentifier, siteCode, clinicalServiceId, institutionCode, EligibilityCheckWorker.DIRECT_ELIGIBILITY_TYPE);
    }

    /**
     * This REST service is used to verify that a patient has visited a clinic by type of care in the past configured request eligibility days.
     *
     * @param siteCode
     * @param patientId
     * @param assigningAuthority
     * @param clinicalServiceId
     * @return PatientVisited
     */
    @GET
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    @Path("/site/{site-code}/patient/{assigning-authority}/{patient-id}/request-eligibility/visited-in-past-months")
    @JwtResourceRestricted
    public PatientVisited getVisitedInPastMonthsByTypeOfCareRequestEligibility(@NotNull @PathParam("site-code") String siteCode,
                                                                                @NotNull @PathParam("assigning-authority") String assigningAuthority,
                                                                                @NotNull @PathParam("patient-id") String patientId,
                                                                                @NotNull @QueryParam("clinical-service") final String clinicalServiceId,
                                                                                @NotNull @QueryParam("institution-code") final String institutionCode) {

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        return eligibilityCheckWorker.hasVisitedClinicInPastMonthsByTypeOfCare(patientIdentifier, siteCode, clinicalServiceId, institutionCode, EligibilityCheckWorker.REQUEST_ELIGIBILITY_TYPE);
    }

    /**
     * This REST service is used to return the patient pact team (primary & team list: reference line 75 of EligibilityCheckWorker) information.
     *
     * @param siteCode
     * @param patientId
     * @param assigningAuthority
     * @return FacilityMemberTeam
     */
    @GET
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    @Path("/site/{site-code}/patient/{assigning-authority}/{patient-id}/pact-team")
    @JwtResourceRestricted
    public Collection<FacilityMemberTeam> getPactTeamMembers(@NotNull @PathParam("site-code") String siteCode,
                                                             @NotNull @PathParam("assigning-authority") String assigningAuthority,
                                                             @NotNull @PathParam("patient-id") String patientId) {

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        return dsFacilityProviderDataService.getTeamMembers(patientIdentifier, siteCode, CoreSettings.PRIMARY_CARE_ID, TeamListCodes.PACT_TEAM);
    }

    /**
     * This REST service is used to verify if patient request limit reached.
     *
     * @param patientId
     * @param assigningAuthority
     * @return PatientRequestLimit
     */
    @GET
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    @Path("/patient/{assigning-authority}/{patient-id}/request-limit")
    @JwtResourceRestricted
    public PatientRequestLimit getPatientMetRequestLimit(@NotNull @PathParam("assigning-authority") String assigningAuthority,
                                                         @NotNull @PathParam("patient-id") String patientId,
                                                         @NotNull @QueryParam("clinical-service") final String clinicalServiceId,
                                                         @NotNull @QueryParam("institution-code") String institutionCode) {

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        return appointmentRequestDataService.getPatientRequestLimit(patientIdentifier, clinicalServiceId, institutionCode);
    }

    /**
     * This REST service is used to return custom text message given institution code.
     *
     * @param institutionCode
     * @return CustomFriendlyText
     */
    @GET
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    @Path("/custom-friendly-text")
    @JwtRbacRestricted("Veteran")
    public CustomFriendlyText getCustomFriendlyText(@NotNull @QueryParam("institution-code") String institutionCode) {
        return customFriendlyTextService.fetchByInstitutionCode(institutionCode);
    }

    /**
     * This REST service is used to return the provider information by type of care.
     *
     * @param siteCode
     * @param assigningAuthority
     * @param patientId
     * @return FacilityMemberTeam
     */
    @GET
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    @Path("/site/{site-code}/patient/{assigning-authority}/{patient-id}/providers")
    @JwtResourceRestricted
    public Collection<FacilityMemberTeam> getProviders(@NotNull @PathParam("site-code") String siteCode,
                                                       @NotNull @PathParam("assigning-authority") String assigningAuthority,
                                                       @NotNull @PathParam("patient-id") String patientId) {

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        return dsFacilityProviderDataService.getTeamMembers(patientIdentifier, siteCode, CoreSettings.PRIMARY_CARE_ID, TeamListCodes.PROVIDERS);
    }

    /**
     * This REST service is used to return the team list information.
     *
     * @param siteCode
     * @param assigningAuthority
     * @param patientId
     * @return FacilityMemberTeam
     */
    @GET
    @Produces({MediaType.APPLICATION_XML, MediaType.APPLICATION_JSON})
    @Path("/site/{site-code}/patient/{assigning-authority}/{patient-id}/team-list")
    @JwtResourceRestricted
    public Collection<FacilityMemberTeam> getTeamMembers(@NotNull @PathParam("site-code") String siteCode,
                                                         @NotNull @PathParam("assigning-authority") String assigningAuthority,
                                                         @NotNull @PathParam("patient-id") String patientId) {

        PatientIdentifier patientIdentifier = new PatientIdentifier(assigningAuthority, patientId);

        return dsFacilityProviderDataService.getTeamMembers(patientIdentifier, siteCode, CoreSettings.PRIMARY_CARE_ID, TeamListCodes.TEAM_LIST);
    }

}